
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
% FILE: sphere_diff_B2S.M   
% Sphere with imposed temperature at external boundary
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

addpath(genpath('.'))

clear all

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% user input
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

R=1;                                         % outer radius
NE=5;                                        % number of elements
rhoc=2.;                                     % capacity
k=1.;                                        % conductivity
TD=1;                                        % value of temperature in r=R
tf=1;                                        % final time
Dt=.02;                                      % time step
implicit=0;                                  % implicit or not? 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% pre-processor
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

h=R/NE;                                      % size of one element
NN=NE+1;                                     % number of nodes
coor=[0:h:R]';                               % nodal coordinates: uniform discretization 
neq=NN-1;                                    % number of equations
temp=zeros(NN,1);                            % initialisation of temperatures
temp(NN)=TD;                                 % boundary conditions
dof=[1:1:NN-1 -NN];                          % degress of freedom

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% assemblage of matrices
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

C=zeros(neq,neq);                            % allocates conductivity matrix
if implicit==0,
  M=zeros(neq,1);                            % allocates condensed capacity "matrix" (vector)
else 
  M=zeros(neq,neq);                          % allocates capacity matrix
end 
F=zeros(neq,1);                              % allocates rhs side for diffusion part

for e=1:NE,                                  % loop over elements to assemble matrices
  S=[coor(e) coor(e+1)];                     % creates segment
  dofe=dof(e:e+1);
  pe=find(dofe>0);
  Ie=dofe(pe);                               % gets value of associated DOFs 
  if implicit==0,
    Me=caplump_B2S(S,rhoc);                  % element capacity diagonal matrix
    M(Ie)=M(Ie)+Me(pe);                      % assemblage of capacity diagonal matrix
  else 
    Me=cap_B2S(S,rhoc);                      % element capacity consistent matrix
    M(Ie,Ie)=M(Ie,Ie)+Me(pe,pe);             % assemblage of capacity consistent matrix
  end 
  Ce=cond_B2S(S,k);                          % element conductivity matrix
  C(Ie,Ie)=C(Ie,Ie)+Ce(pe,pe);               % assemblage of conductivity matrix
  pe_TDe=find(dofe<0);
  Ie_TDe=-dofe(pe_TDe);
  F(Ie)=F(Ie)-Ce(pe,pe_TDe)*temp(Ie_TDe);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% time marching solution
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if implicit==0,                              % computes stable  Dt for explicit option
  uu=eigs(C,sparse(diag(M)),1);
  stable_time_increment=2/uu
end  

if implicit==1,                              % prepares Cholesky decomposition
  C=C+M/Dt;                                 
  RC=chol(C);
  clear C
end

ir=find(dof>0);                              % finds active dofs
eq=dof(ir);
T=zeros(neq,1);
nstep=floor(tf/Dt);                          % number of time steps assuming constant dt
vec_time=zeros(nstep,1);                     % vector for time values (for post-process.)
vec_temperature=zeros(nstep,1);              % vector for cavity temperatures

%-----------------------------------------------------------------------------------------
figmess=figure('Units','normalized','MenuBar','none','NumberTitle','off', ...
               'Position',[.15 .80 .50 .15],'Color','white');
ax1=axes('Visible','off');
t1=text(0,.5,['Total number of steps: ' num2str(nstep)],'Fontsize',13,...
        'HorizontalAlignment','left','Color','b'); pause(.1);
t2=text(0.,.2,'Actual step: 0','Fontsize',13,...
        'HorizontalAlignment','left','Color','r');  pause(.01);
%-----------------------------------------------------------------------------------------

for step=1:nstep,                            % loop over time steps

%-----------------------------------------------------------------------------------------
  set(t2,'string',['Actual step: ' ,num2str(step), '  Node temp: ' ,num2str(temp(1))]); 
		pause(.01);
%-----------------------------------------------------------------------------------------

  if implicit==1,
    Fg=F+1/Dt*M*T;                           % consistent
    T=RC\(RC'\Fg);
  else
    Fg=F-C*T;
    T=T+Dt*Fg./M;                            % updates temperature
		end 
  temp(ir)=T(eq);                            % actual temperatures on all nodes
  vec_time(step)=step*Dt;
  vec_temperature(step)=temp(1);
end

clear RC C M ir eq

close(figmess);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% plots solution and compares with exact one
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

bkgcol=.93*[1 1 1];
figure('Color',bkgcol)                       % plots numerical and exact radial displ.
%axis square
plot(coor,temp)
title('Temperature along the radius at final time')
xlabel('Radial coordinate')
ylabel('Temperature')
set(gca,'Color','w')                         % sets color of current axes

bkgcol=.93*[1 1 1];
figure('Color',bkgcol)                       % plots numerical and exact radial displ.
axis([min(vec_time) max(vec_time) ...
      min(vec_temperature) max(vec_temperature)]);
axis square
axis on
plot(vec_time,vec_temperature,'--')
lambda=k/rhoc;
nn=100;
for step=1:nstep,
  time=vec_time(step);
  an=(1:nn);
  vn=exp(-((an*pi/R).^2)*lambda*time)*2;     % The exact solution is an infinite series...
  an=(-1).^an;                               % here truncated to nn terms 
  vec2(step,1)=an*vn'+1;
end
hold on
plot(vec_time,vec2)
title('Temperature history at sphere center')
xlabel('Time')
ylabel('Temperature')
legend('FEM','exact') 
set(gca,'Color','w')                         % sets color of current axes


